function [grad] = gradient_f(func, para, d, varargin)
% The function computes numerical gradient based on central difference.
%=====================================================================================================
%  INPUT  : func    : function whose first argument is para that returns a scalar or vector of values (nx1)
%           para    : vector of parameters at which gradients are to be computed (px1)
%           d       : increment/delta (a small number) used in calculating the gradients. 
%           varargin: Optional arguments passed to the function
%
%  OUTPUT : grad: matrices of gradients (nxp)
%           the ith colunm of grad (i in [1,p]) = [func(para + e*d) - func(para - e*d)]/(2*d)
%           where e is the ith column of the pxp identity matrix.
%           i.e. change each element of para by some small quantity at each time
%=====================================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 
  
if size(para,1) == 1
    para = para';
end
p = length(para);

try
    y = feval(func,para,varargin{:});
catch FE
    errtxt=['There was an error evaluating the function.  Please check the arguments.  The specific error was:' FE.message];
    error(errtxt);
end

if ~isvector(y)
    error('Output of function ''func'' must be either a scalar or a vector.')
end

% Compute the stepsize/delta/increment (d)
if isempty(d)
    d = eps^(1/3)*max(abs(para),1e-1);
else
    d = d.*max(abs(para),1e-1);
end

n = length(y);

% Gradient matrix
grad = nan(n, p);

ee = sparse(1:p,1:p,d,p,p);

for i = 1:p
    grad(:, i) = (feval(func,para+ee(:,i),varargin{:}) - ...
        feval(func,para-ee(:,i),varargin{:}))/(2*d(i));
end

